/*
 * Decompiled with CFR 0.152.
 */
package org.tribuo.math.la;

import com.google.protobuf.Any;
import com.google.protobuf.ByteString;
import com.google.protobuf.InvalidProtocolBufferException;
import com.google.protobuf.Message;
import com.oracle.labs.mlrg.olcut.util.SortUtil;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.DoubleBuffer;
import java.util.Arrays;
import java.util.List;
import java.util.NoSuchElementException;
import java.util.Objects;
import java.util.Optional;
import java.util.function.DoubleUnaryOperator;
import java.util.logging.Logger;
import java.util.stream.Collectors;
import org.tribuo.math.la.DenseSparseMatrix;
import org.tribuo.math.la.DenseVector;
import org.tribuo.math.la.Matrix;
import org.tribuo.math.la.MatrixIterator;
import org.tribuo.math.la.MatrixTuple;
import org.tribuo.math.la.SGDVector;
import org.tribuo.math.la.SparseVector;
import org.tribuo.math.la.Tensor;
import org.tribuo.math.la.VectorTuple;
import org.tribuo.math.protos.DenseTensorProto;
import org.tribuo.math.protos.TensorProto;
import org.tribuo.math.util.VectorNormalizer;
import org.tribuo.util.Util;

public class DenseMatrix
implements Matrix {
    private static final long serialVersionUID = 1L;
    private static final Logger logger = Logger.getLogger(DenseMatrix.class.getName());
    public static final int CURRENT_VERSION = 0;
    public static final double FACTORIZATION_TOLERANCE = 1.0E-14;
    private static final double DELTA = 1.0E-10;
    protected final double[][] values;
    protected final int dim1;
    protected final int dim2;
    private final int[] shape;
    private final int numElements;

    public DenseMatrix(int dim1, int dim2) {
        this.values = new double[dim1][dim2];
        this.dim1 = dim1;
        this.dim2 = dim2;
        this.shape = new int[]{dim1, dim2};
        this.numElements = dim1 * dim2;
    }

    public DenseMatrix(DenseMatrix other) {
        this.values = new double[other.values.length][];
        for (int i = 0; i < this.values.length; ++i) {
            this.values[i] = new double[other.values[i].length];
            for (int j = 0; j < this.values[i].length; ++j) {
                this.values[i][j] = other.get(i, j);
            }
        }
        this.dim1 = other.dim1;
        this.dim2 = other.dim2;
        this.shape = new int[]{this.dim1, this.dim2};
        this.numElements = this.dim1 * this.dim2;
    }

    public DenseMatrix(Matrix other) {
        this.dim1 = other.getDimension1Size();
        this.dim2 = other.getDimension2Size();
        this.values = new double[this.dim1][this.dim2];
        if (other instanceof DenseMatrix) {
            for (int i = 0; i < this.dim1; ++i) {
                for (int j = 0; j < this.dim2; ++j) {
                    this.values[i][j] = other.get(i, j);
                }
            }
        } else {
            for (MatrixTuple t : other) {
                this.values[t.i][t.j] = t.value;
            }
        }
        this.shape = new int[]{this.dim1, this.dim2};
        this.numElements = this.dim1 * this.dim2;
    }

    DenseMatrix(double[][] values) {
        this.values = values;
        this.dim1 = values.length;
        this.dim2 = values[0].length;
        this.shape = new int[]{this.dim1, this.dim2};
        this.numElements = this.dim1 * this.dim2;
    }

    public static DenseMatrix createDenseMatrix(double[][] values) {
        double[][] newValues = new double[values.length][];
        int sizeCounter = -1;
        for (int i = 0; i < newValues.length; ++i) {
            if (sizeCounter == -1) {
                sizeCounter = values[i].length;
            }
            if (sizeCounter != values[i].length) {
                throw new IllegalArgumentException("DenseMatrix must not be ragged. Expected dim2 = " + sizeCounter + ", but found " + values[i].length + " at index " + i);
            }
            newValues[i] = Arrays.copyOf(values[i], values[i].length);
        }
        return new DenseMatrix(newValues);
    }

    public static DenseMatrix createDenseMatrix(SGDVector[] vectors) {
        if (vectors == null || vectors.length == 0) {
            throw new IllegalArgumentException("Invalid vector array.");
        }
        double[][] newValues = new double[vectors.length][];
        int size = vectors[0].size();
        for (int i = 0; i < vectors.length; ++i) {
            if (vectors[i].size() != size) {
                throw new IllegalArgumentException("Expected size " + size + " but found size " + vectors[i].size() + " at index " + i);
            }
            newValues[i] = vectors[i].toArray();
        }
        return new DenseMatrix(newValues);
    }

    public static DenseMatrix deserializeFromProto(int version, String className, Any message) throws InvalidProtocolBufferException {
        if (version < 0 || version > 0) {
            throw new IllegalArgumentException("Unknown version " + version + ", this class supports at most version " + 0);
        }
        DenseTensorProto proto = (DenseTensorProto)message.unpack(DenseTensorProto.class);
        return DenseMatrix.unpackProto(proto);
    }

    protected static DenseMatrix unpackProto(DenseTensorProto proto) {
        List<Integer> shapeList = proto.getDimensionsList();
        int[] shape = Util.toPrimitiveInt(shapeList);
        if (shape.length != 2) {
            throw new IllegalArgumentException("Invalid proto, expected a matrix, found shape " + Arrays.toString(shape));
        }
        for (int i = 0; i < shape.length; ++i) {
            if (shape[i] >= 1) continue;
            throw new IllegalArgumentException("Invalid proto, shape must be positive, found " + shape[i] + " at position " + i);
        }
        int numElements = Util.product((int[])shape);
        DoubleBuffer buffer = proto.getValues().asReadOnlyByteBuffer().order(ByteOrder.LITTLE_ENDIAN).asDoubleBuffer();
        if (buffer.remaining() != numElements) {
            throw new IllegalArgumentException("Invalid proto, claimed " + numElements + ", but only had " + buffer.remaining() + " values");
        }
        double[][] values = new double[shape[0]][shape[1]];
        for (int i = 0; i < values.length; ++i) {
            buffer.get(values[i]);
        }
        return new DenseMatrix(values);
    }

    public TensorProto serialize() {
        TensorProto.Builder builder = TensorProto.newBuilder();
        builder.setVersion(0);
        builder.setClassName(DenseMatrix.class.getName());
        DenseTensorProto.Builder dataBuilder = DenseTensorProto.newBuilder();
        dataBuilder.addAllDimensions(Arrays.stream(this.shape).boxed().collect(Collectors.toList()));
        ByteBuffer buffer = ByteBuffer.allocate(this.numElements * 8).order(ByteOrder.LITTLE_ENDIAN);
        DoubleBuffer doubleBuffer = buffer.asDoubleBuffer();
        for (int i = 0; i < this.values.length; ++i) {
            doubleBuffer.put(this.values[i]);
        }
        doubleBuffer.rewind();
        dataBuilder.setValues(ByteString.copyFrom((ByteBuffer)buffer));
        builder.setSerializedData(Any.pack((Message)dataBuilder.build()));
        return builder.build();
    }

    @Override
    public int[] getShape() {
        return this.shape;
    }

    @Override
    public Tensor reshape(int[] newShape) {
        int sum = Tensor.shapeSum(newShape);
        if (sum != this.numElements) {
            throw new IllegalArgumentException("Invalid shape " + Arrays.toString(newShape) + ", expected something with " + this.numElements + " elements.");
        }
        if (newShape.length == 2) {
            DenseMatrix matrix = new DenseMatrix(newShape[0], newShape[1]);
            for (int a = 0; a < this.numElements; ++a) {
                int oldI = a % this.dim1;
                int oldJ = a % this.dim2;
                int i = a % newShape[0];
                int j = a / newShape[0];
                matrix.set(i, j, this.get(oldI, oldJ));
            }
            return matrix;
        }
        if (newShape.length == 1) {
            DenseVector vector = new DenseVector(this.numElements);
            int a = 0;
            for (int i = 0; i < this.dim1; ++i) {
                for (int j = 0; j < this.dim2; ++j) {
                    vector.set(a, this.get(i, j));
                    ++a;
                }
            }
            return vector;
        }
        throw new IllegalArgumentException("Only supports 1 or 2 dimensional tensors.");
    }

    @Override
    public DenseMatrix copy() {
        return new DenseMatrix(this);
    }

    @Override
    public double get(int i, int j) {
        return this.values[i][j];
    }

    public DenseVector gatherAcrossDim1(int[] elements) {
        if (elements.length != this.dim2) {
            throw new IllegalArgumentException("Invalid number of elements to gather, must select one per value of dim2");
        }
        double[] outputValues = new double[this.dim2];
        for (int i = 0; i < elements.length; ++i) {
            outputValues[i] = this.get(elements[i], i);
        }
        return new DenseVector(outputValues);
    }

    public DenseVector gatherAcrossDim2(int[] elements) {
        if (elements.length != this.dim1) {
            throw new IllegalArgumentException("Invalid number of elements to gather, must select one per value of dim1");
        }
        double[] outputValues = new double[this.dim1];
        for (int i = 0; i < elements.length; ++i) {
            outputValues[i] = this.get(i, elements[i]);
        }
        return new DenseVector(outputValues);
    }

    public DenseMatrix transpose() {
        double[][] newValues = new double[this.dim2][this.dim1];
        for (int i = 0; i < this.dim1; ++i) {
            for (int j = 0; j < this.dim2; ++j) {
                newValues[j][i] = this.get(i, j);
            }
        }
        return new DenseMatrix(newValues);
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (!(o instanceof DenseMatrix)) {
            return false;
        }
        DenseMatrix that = (DenseMatrix)o;
        if (this.dim1 == that.dim1 && this.dim2 == that.dim2 && this.numElements == that.numElements && Arrays.equals(this.getShape(), that.getShape())) {
            for (int i = 0; i < this.dim1; ++i) {
                for (int j = 0; j < this.dim2; ++j) {
                    if (!(Math.abs(this.get(i, j) - that.get(i, j)) > 1.0E-10)) continue;
                    return false;
                }
            }
            return true;
        }
        return false;
    }

    public int hashCode() {
        int result = Objects.hash(this.dim1, this.dim2, this.numElements);
        result = 31 * result + Arrays.deepHashCode((Object[])this.values);
        result = 31 * result + Arrays.hashCode(this.getShape());
        return result;
    }

    @Override
    public void set(int i, int j, double value) {
        this.values[i][j] = value;
    }

    @Override
    public int getDimension1Size() {
        return this.dim1;
    }

    @Override
    public int getDimension2Size() {
        return this.dim2;
    }

    @Override
    public DenseVector leftMultiply(SGDVector input) {
        if (input.size() == this.dim2) {
            double[] output = new double[this.dim1];
            if (input instanceof DenseVector) {
                for (int i = 0; i < this.dim1; ++i) {
                    for (int j = 0; j < this.dim2; ++j) {
                        int n = i;
                        output[n] = output[n] + this.get(i, j) * input.get(j);
                    }
                }
            } else {
                for (VectorTuple tuple : input) {
                    for (int i = 0; i < output.length; ++i) {
                        int n = i;
                        output[n] = output[n] + this.get(i, tuple.index) * tuple.value;
                    }
                }
            }
            return new DenseVector(output);
        }
        throw new IllegalArgumentException("input.size() != dim2, input.size() = " + input.size() + ", dim1,dim2 = " + this.dim1 + "," + this.dim2);
    }

    @Override
    public DenseVector rightMultiply(SGDVector input) {
        if (input.size() == this.dim1) {
            double[] output = new double[this.dim2];
            if (input instanceof DenseVector) {
                for (int i = 0; i < this.dim1; ++i) {
                    double curValue = input.get(i);
                    for (int j = 0; j < this.dim2; ++j) {
                        int n = j;
                        output[n] = output[n] + this.get(i, j) * curValue;
                    }
                }
            } else {
                for (VectorTuple tuple : input) {
                    for (int i = 0; i < output.length; ++i) {
                        int n = i;
                        output[n] = output[n] + this.get(tuple.index, i) * tuple.value;
                    }
                }
            }
            return new DenseVector(output);
        }
        throw new IllegalArgumentException("input.size() != dim1");
    }

    @Override
    public DenseMatrix matrixMultiply(Matrix other) {
        if (this.dim2 == other.getDimension1Size()) {
            if (other instanceof DenseMatrix) {
                DenseMatrix otherDense = (DenseMatrix)other;
                double[][] output = new double[this.dim1][otherDense.dim2];
                for (int i = 0; i < this.dim1; ++i) {
                    for (int j = 0; j < otherDense.dim2; ++j) {
                        output[i][j] = this.columnRowDot(i, j, otherDense);
                    }
                }
                return new DenseMatrix(output);
            }
            if (other instanceof DenseSparseMatrix) {
                DenseSparseMatrix otherSparse = (DenseSparseMatrix)other;
                int otherDim2 = otherSparse.getDimension2Size();
                double[][] output = new double[this.dim1][otherDim2];
                for (int i = 0; i < this.dim1; ++i) {
                    for (int j = 0; j < otherDim2; ++j) {
                        output[i][j] = this.columnRowDot(i, j, otherSparse);
                    }
                }
                return new DenseMatrix(output);
            }
            throw new IllegalArgumentException("Unknown matrix type " + other.getClass().getName());
        }
        throw new IllegalArgumentException("Invalid matrix dimensions, this.shape=" + Arrays.toString(this.shape) + ", other.shape = " + Arrays.toString(other.getShape()));
    }

    @Override
    public DenseMatrix matrixMultiply(Matrix other, boolean transposeThis, boolean transposeOther) {
        if (transposeThis && transposeOther) {
            return this.matrixMultiplyTransposeBoth(other);
        }
        if (transposeThis) {
            return this.matrixMultiplyTransposeThis(other);
        }
        if (transposeOther) {
            return this.matrixMultiplyTransposeOther(other);
        }
        return this.matrixMultiply(other);
    }

    private DenseMatrix matrixMultiplyTransposeBoth(Matrix other) {
        if (this.dim1 == other.getDimension2Size()) {
            if (other instanceof DenseMatrix) {
                DenseMatrix otherDense = (DenseMatrix)other;
                double[][] output = new double[this.dim2][otherDense.dim1];
                for (int i = 0; i < this.dim2; ++i) {
                    for (int j = 0; j < otherDense.dim1; ++j) {
                        output[i][j] = this.rowColumnDot(i, j, otherDense);
                    }
                }
                return new DenseMatrix(output);
            }
            if (other instanceof DenseSparseMatrix) {
                DenseSparseMatrix otherSparse = (DenseSparseMatrix)other;
                int otherDim1 = otherSparse.getDimension1Size();
                double[][] output = new double[this.dim2][otherDim1];
                for (int i = 0; i < this.dim2; ++i) {
                    for (int j = 0; j < otherDim1; ++j) {
                        output[i][j] = this.rowColumnDot(i, j, otherSparse);
                    }
                }
                return new DenseMatrix(output);
            }
            throw new IllegalArgumentException("Unknown matrix type " + other.getClass().getName());
        }
        throw new IllegalArgumentException("Invalid matrix dimensions, this.shape=" + Arrays.toString(this.shape) + ", other.shape = " + Arrays.toString(other.getShape()));
    }

    private DenseMatrix matrixMultiplyTransposeThis(Matrix other) {
        if (this.dim1 == other.getDimension1Size()) {
            if (other instanceof DenseMatrix) {
                DenseMatrix otherDense = (DenseMatrix)other;
                double[][] output = new double[this.dim2][otherDense.dim2];
                for (int i = 0; i < this.dim2; ++i) {
                    for (int j = 0; j < otherDense.dim2; ++j) {
                        output[i][j] = this.columnColumnDot(i, j, otherDense);
                    }
                }
                return new DenseMatrix(output);
            }
            if (other instanceof DenseSparseMatrix) {
                DenseSparseMatrix otherSparse = (DenseSparseMatrix)other;
                int otherDim2 = otherSparse.getDimension2Size();
                double[][] output = new double[this.dim2][otherDim2];
                for (int i = 0; i < this.dim2; ++i) {
                    for (int j = 0; j < otherDim2; ++j) {
                        output[i][j] = this.columnColumnDot(i, j, otherSparse);
                    }
                }
                return new DenseMatrix(output);
            }
            throw new IllegalArgumentException("Unknown matrix type " + other.getClass().getName());
        }
        throw new IllegalArgumentException("Invalid matrix dimensions, this.shape=" + Arrays.toString(this.shape) + ", other.shape = " + Arrays.toString(other.getShape()));
    }

    private DenseMatrix matrixMultiplyTransposeOther(Matrix other) {
        if (this.dim2 == other.getDimension2Size()) {
            if (other instanceof DenseMatrix) {
                DenseMatrix otherDense = (DenseMatrix)other;
                double[][] output = new double[this.dim1][otherDense.dim1];
                for (int i = 0; i < this.dim1; ++i) {
                    for (int j = 0; j < otherDense.dim1; ++j) {
                        output[i][j] = this.rowRowDot(i, j, otherDense);
                    }
                }
                return new DenseMatrix(output);
            }
            if (other instanceof DenseSparseMatrix) {
                DenseSparseMatrix otherSparse = (DenseSparseMatrix)other;
                int otherDim1 = otherSparse.getDimension1Size();
                double[][] output = new double[this.dim1][otherDim1];
                for (int i = 0; i < this.dim1; ++i) {
                    for (int j = 0; j < otherDim1; ++j) {
                        output[i][j] = this.rowRowDot(i, j, otherSparse);
                    }
                }
                return new DenseMatrix(output);
            }
            throw new IllegalArgumentException("Unknown matrix type " + other.getClass().getName());
        }
        throw new IllegalArgumentException("Invalid matrix dimensions, this.shape=" + Arrays.toString(this.shape) + ", other.shape = " + Arrays.toString(other.getShape()));
    }

    private double columnRowDot(int rowIndex, int otherColIndex, Matrix other) {
        double sum = 0.0;
        for (int i = 0; i < this.dim2; ++i) {
            sum += this.get(rowIndex, i) * other.get(i, otherColIndex);
        }
        return sum;
    }

    private double rowColumnDot(int colIndex, int otherRowIndex, Matrix other) {
        double sum = 0.0;
        for (int i = 0; i < this.dim1; ++i) {
            sum += this.get(i, colIndex) * other.get(otherRowIndex, i);
        }
        return sum;
    }

    private double columnColumnDot(int colIndex, int otherColIndex, Matrix other) {
        double sum = 0.0;
        for (int i = 0; i < this.dim1; ++i) {
            sum += this.get(i, colIndex) * other.get(i, otherColIndex);
        }
        return sum;
    }

    private double rowRowDot(int rowIndex, int otherRowIndex, Matrix other) {
        double sum = 0.0;
        for (int i = 0; i < this.dim2; ++i) {
            sum += this.get(rowIndex, i) * other.get(otherRowIndex, i);
        }
        return sum;
    }

    @Override
    public DenseVector rowSum() {
        double[] rowSum = new double[this.dim1];
        for (int i = 0; i < this.dim1; ++i) {
            double tmp = 0.0;
            for (int j = 0; j < this.dim2; ++j) {
                tmp += this.get(i, j);
            }
            rowSum[i] = tmp;
        }
        return new DenseVector(rowSum);
    }

    @Override
    public void rowScaleInPlace(DenseVector scalingCoefficients) {
        for (int i = 0; i < this.dim1; ++i) {
            double scalar = scalingCoefficients.get(i);
            int j = 0;
            while (j < this.dim2) {
                double[] dArray = this.values[i];
                int n = j++;
                dArray[n] = dArray[n] * scalar;
            }
        }
    }

    @Override
    public void add(int i, int j, double value) {
        double[] dArray = this.values[i];
        int n = j;
        dArray[n] = dArray[n] + value;
    }

    public void addAcrossDim1(int[] indices, double value) {
        if (indices.length != this.dim2) {
            throw new IllegalArgumentException("Invalid number of elements to add, must select one per value of dim2");
        }
        int i = 0;
        while (i < indices.length) {
            double[] dArray = this.values[indices[i]];
            int n = i++;
            dArray[n] = dArray[n] + value;
        }
    }

    public void addAcrossDim2(int[] indices, double value) {
        if (indices.length != this.dim1) {
            throw new IllegalArgumentException("Invalid number of elements to indices, must select one per value of dim1");
        }
        for (int i = 0; i < indices.length; ++i) {
            double[] dArray = this.values[i];
            int n = indices[i];
            dArray[n] = dArray[n] + value;
        }
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    @Override
    public void intersectAndAddInPlace(Tensor other, DoubleUnaryOperator f) {
        if (!(other instanceof Matrix)) throw new IllegalArgumentException("Adding a non-Matrix to a Matrix");
        Matrix otherMat = (Matrix)other;
        if (this.dim1 != otherMat.getDimension1Size() || this.dim2 != otherMat.getDimension2Size()) throw new IllegalArgumentException("Matrices are not the same size, this(" + this.dim1 + "," + this.dim2 + "), other(" + otherMat.getDimension1Size() + "," + otherMat.getDimension2Size() + ")");
        if (otherMat instanceof DenseMatrix) {
            for (int i = 0; i < this.dim1; ++i) {
                for (int j = 0; j < this.dim2; ++j) {
                    double[] dArray = this.values[i];
                    int n = j;
                    dArray[n] = dArray[n] + f.applyAsDouble(otherMat.get(i, j));
                }
            }
            return;
        } else {
            for (MatrixTuple tuple : otherMat) {
                double[] dArray = this.values[tuple.i];
                int n = tuple.j;
                dArray[n] = dArray[n] + f.applyAsDouble(tuple.value);
            }
        }
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    @Override
    public void hadamardProductInPlace(Tensor other, DoubleUnaryOperator f) {
        if (!(other instanceof Matrix)) throw new IllegalArgumentException("Adding a non-Matrix to a Matrix");
        Matrix otherMat = (Matrix)other;
        if (this.dim1 != otherMat.getDimension1Size() || this.dim2 != otherMat.getDimension2Size()) throw new IllegalArgumentException("Matrices are not the same size, this(" + this.dim1 + "," + this.dim2 + "), other(" + otherMat.getDimension1Size() + "," + otherMat.getDimension2Size() + ")");
        if (otherMat instanceof DenseMatrix) {
            for (int i = 0; i < this.dim1; ++i) {
                for (int j = 0; j < this.dim2; ++j) {
                    double[] dArray = this.values[i];
                    int n = j;
                    dArray[n] = dArray[n] * f.applyAsDouble(otherMat.get(i, j));
                }
            }
            return;
        } else {
            for (MatrixTuple tuple : otherMat) {
                double[] dArray = this.values[tuple.i];
                int n = tuple.j;
                dArray[n] = dArray[n] * f.applyAsDouble(tuple.value);
            }
        }
    }

    @Override
    public void foreachInPlace(DoubleUnaryOperator f) {
        for (int i = 0; i < this.values.length; ++i) {
            for (int j = 0; j < this.dim2; ++j) {
                this.values[i][j] = f.applyAsDouble(this.values[i][j]);
            }
        }
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    public void broadcastIntersectAndAddInPlace(SGDVector input, boolean broadcastOverDim1) {
        if (input instanceof DenseVector) {
            if (broadcastOverDim1) {
                if (input.size() != this.dim2) throw new IllegalArgumentException("Input vector must have dimension equal to dim 2, input.size() = " + input.size() + ", dim2 = " + this.dim2);
                for (int i = 0; i < this.dim1; ++i) {
                    for (int j = 0; j < this.dim2; ++j) {
                        double[] dArray = this.values[i];
                        int n = j;
                        dArray[n] = dArray[n] + input.get(j);
                    }
                }
                return;
            } else {
                if (input.size() != this.dim1) throw new IllegalArgumentException("Input vector must have dimension equal to dim 1, input.size() = " + input.size() + ", dim1 = " + this.dim1);
                for (int i = 0; i < this.dim1; ++i) {
                    double ith = input.get(i);
                    int j = 0;
                    while (j < this.dim2) {
                        double[] dArray = this.values[i];
                        int n = j++;
                        dArray[n] = dArray[n] + ith;
                    }
                }
            }
            return;
        } else {
            if (!(input instanceof SparseVector)) throw new IllegalArgumentException("Input vector was neither dense nor sparse.");
            if (broadcastOverDim1) {
                if (input.size() != this.dim2) throw new IllegalArgumentException("Input vector must have dimension equal to dim 2, input.size() = " + input.size() + ", dim2 = " + this.dim2);
                for (int i = 0; i < this.dim1; ++i) {
                    for (VectorTuple v : input) {
                        double[] dArray = this.values[i];
                        int n = v.index;
                        dArray[n] = dArray[n] + v.value;
                    }
                }
                return;
            } else {
                if (input.size() != this.dim1) throw new IllegalArgumentException("Input vector must have dimension equal to dim 1, input.size() = " + input.size() + ", dim1 = " + this.dim1);
                for (VectorTuple v : input) {
                    int j = 0;
                    while (j < this.dim2) {
                        double[] dArray = this.values[v.index];
                        int n = j++;
                        dArray[n] = dArray[n] + v.value;
                    }
                }
            }
        }
    }

    @Override
    public int numActiveElements(int row) {
        return this.dim2;
    }

    @Override
    public DenseVector getRow(int i) {
        if (i < 0 || i > this.dim1) {
            throw new IllegalArgumentException("Invalid row index, must be [0," + this.dim1 + "), received " + i);
        }
        return new DenseVector(this.values[i]);
    }

    @Override
    public DenseVector getColumn(int index) {
        if (index < 0 || index > this.dim2) {
            throw new IllegalArgumentException("Invalid column index, must be [0," + this.dim2 + "), received " + index);
        }
        double[] output = new double[this.dim1];
        for (int i = 0; i < this.dim1; ++i) {
            output[i] = this.get(i, index);
        }
        return new DenseVector(output);
    }

    public void setColumn(int index, SGDVector vector) {
        if (index < 0 || index > this.dim2) {
            throw new IllegalArgumentException("Invalid column index, must be [0," + this.dim2 + "), received " + index);
        }
        if (vector.size() == this.dim1) {
            if (vector instanceof DenseVector) {
                for (int i = 0; i < this.dim1; ++i) {
                    this.values[i][index] = vector.get(i);
                }
            } else {
                for (VectorTuple t : vector) {
                    this.values[t.index][index] = t.value;
                }
            }
        } else {
            throw new IllegalArgumentException("Vector size mismatch, expected " + this.dim1 + " found " + vector.size());
        }
    }

    public double rowSum(int rowIndex) {
        if (rowIndex < 0 || rowIndex > this.dim1) {
            throw new IllegalArgumentException("Invalid row index, must be [0," + this.dim1 + "), received " + rowIndex);
        }
        double sum = 0.0;
        for (int i = 0; i < this.dim2; ++i) {
            sum += this.get(rowIndex, i);
        }
        return sum;
    }

    public double columnSum(int columnIndex) {
        if (columnIndex < 0 || columnIndex > this.dim2) {
            throw new IllegalArgumentException("Invalid column index, must be [0," + this.dim2 + "), received " + columnIndex);
        }
        double sum = 0.0;
        for (int i = 0; i < this.dim1; ++i) {
            sum += this.get(i, columnIndex);
        }
        return sum;
    }

    @Override
    public double twoNorm() {
        double output = 0.0;
        for (int i = 0; i < this.dim1; ++i) {
            for (int j = 0; j < this.dim2; ++j) {
                double value = this.get(i, j);
                output += value * value;
            }
        }
        return Math.sqrt(output);
    }

    public double[][] toArray() {
        double[][] copy = new double[this.dim1][];
        for (int i = 0; i < this.dim1; ++i) {
            copy[i] = Arrays.copyOf(this.values[i], this.dim2);
        }
        return copy;
    }

    public boolean isSquare() {
        return this.dim1 == this.dim2;
    }

    public boolean isSymmetric() {
        if (!this.isSquare()) {
            return false;
        }
        for (int i = 0; i < this.dim1; ++i) {
            for (int j = i + 1; j < this.dim1; ++j) {
                if (Double.compare(this.get(i, j), this.get(j, i)) == 0) continue;
                return false;
            }
        }
        return true;
    }

    public Optional<CholeskyFactorization> choleskyFactorization() {
        int j;
        int i;
        if (!this.isSymmetric()) {
            logger.fine("Returning empty optional as matrix is not symmetric");
            return Optional.empty();
        }
        DenseMatrix chol = new DenseMatrix(this);
        double[][] cholMatrix = chol.values;
        for (i = 0; i < this.dim1; ++i) {
            for (j = i; j < this.dim1; ++j) {
                double sum = cholMatrix[i][j];
                for (int k = i - 1; k >= 0; --k) {
                    sum -= cholMatrix[i][k] * cholMatrix[j][k];
                }
                if (i == j) {
                    if (sum <= 1.0E-14) {
                        logger.fine("Returning empty optional as matrix is not positive definite");
                        return Optional.empty();
                    }
                    cholMatrix[i][i] = Math.sqrt(sum);
                    continue;
                }
                cholMatrix[j][i] = sum / cholMatrix[i][i];
            }
        }
        for (i = 0; i < this.dim1; ++i) {
            for (j = 0; j < i; ++j) {
                cholMatrix[j][i] = 0.0;
            }
        }
        return Optional.of(new CholeskyFactorization(chol));
    }

    public Optional<LUFactorization> luFactorization() {
        int i;
        int i2;
        if (!this.isSquare()) {
            logger.fine("Returning empty optional as matrix is not square");
            return Optional.empty();
        }
        DenseMatrix lu = new DenseMatrix(this);
        double[][] luMatrix = lu.values;
        int[] permutation = new int[this.dim1];
        boolean oddSwaps = false;
        for (i2 = 0; i2 < this.dim1; ++i2) {
            permutation[i2] = i2;
        }
        for (i2 = 0; i2 < this.dim1; ++i2) {
            double max = 0.0;
            int maxIdx = i2;
            for (int k = i2; k < this.dim1; ++k) {
                double cur = Math.abs(luMatrix[k][i2]);
                if (!(cur > max)) continue;
                max = cur;
                maxIdx = k;
            }
            if (max < 1.0E-14) {
                logger.fine("Returning empty optional as matrix is singular");
                return Optional.empty();
            }
            if (maxIdx != i2) {
                int tmpIdx = permutation[maxIdx];
                permutation[maxIdx] = permutation[i2];
                permutation[i2] = tmpIdx;
                oddSwaps = !oddSwaps;
                double[] tmpRow = luMatrix[maxIdx];
                luMatrix[maxIdx] = luMatrix[i2];
                luMatrix[i2] = tmpRow;
            }
            for (int j = i2 + 1; j < this.dim1; ++j) {
                double[] dArray = luMatrix[j];
                int n = i2;
                dArray[n] = dArray[n] / luMatrix[i2][i2];
                for (int k = i2 + 1; k < this.dim1; ++k) {
                    double[] dArray2 = luMatrix[j];
                    int n2 = k;
                    dArray2[n2] = dArray2[n2] - luMatrix[j][i2] * luMatrix[i2][k];
                }
            }
        }
        DenseMatrix l = new DenseMatrix(lu);
        DenseMatrix u = new DenseMatrix(lu);
        for (i = 0; i < this.dim1; ++i) {
            Arrays.fill(u.values[i], 0, i, 0.0);
        }
        for (i = 0; i < this.dim1; ++i) {
            for (int j = 0; j <= i; ++j) {
                if (i == j) {
                    l.values[i][j] = 1.0;
                    continue;
                }
                l.values[j][i] = 0.0;
            }
        }
        return Optional.of(new LUFactorization(l, u, permutation, oddSwaps));
    }

    public Optional<EigenDecomposition> eigenDecomposition() {
        int k;
        int j;
        if (!this.isSymmetric()) {
            logger.fine("Returning empty optional as matrix is not symmetric");
            return Optional.empty();
        }
        DenseMatrix transform = new DenseMatrix(this);
        double[][] transformValues = transform.values;
        double[] diagonal = new double[this.dim1];
        double[] offDiagonal = new double[this.dim1];
        System.arraycopy(transformValues[this.dim1 - 1], 0, diagonal, 0, this.dim1);
        for (int i = this.dim1 - 1; i > 0; --i) {
            double scale = 0.0;
            for (int k2 = 0; k2 < i; ++k2) {
                scale += Math.abs(diagonal[k2]);
            }
            double diagElement = 0.0;
            if (scale == 0.0) {
                offDiagonal[i] = 0.0;
                for (j = 0; j < i; ++j) {
                    diagonal[j] = transformValues[i - 1][j];
                    transformValues[i][j] = 0.0;
                    transformValues[j][i] = 0.0;
                }
            } else {
                int j2;
                for (k = 0; k < i; ++k) {
                    double tmp = diagonal[k] / scale;
                    diagElement += tmp * tmp;
                    diagonal[k] = tmp;
                    offDiagonal[k] = 0.0;
                }
                double nextDiag = diagonal[i - 1];
                double offDiag = nextDiag >= 0.0 ? -Math.sqrt(diagElement) : Math.sqrt(diagElement);
                offDiagonal[i] = scale * offDiag;
                diagElement -= offDiag * nextDiag;
                diagonal[i - 1] = nextDiag - offDiag;
                for (int j3 = 0; j3 < i; ++j3) {
                    double transDiag;
                    transformValues[j3][i] = transDiag = diagonal[j3];
                    double transOffDiag = offDiagonal[j3] + transformValues[j3][j3] * transDiag;
                    int k3 = j3 + 1;
                    while (k3 < i) {
                        double tmp = transformValues[k3][j3];
                        transOffDiag += tmp * diagonal[k3];
                        int n = k3++;
                        offDiagonal[n] = offDiagonal[n] + tmp * transDiag;
                    }
                    offDiagonal[j3] = transOffDiag;
                }
                double scaledElementSum = 0.0;
                for (int j4 = 0; j4 < i; ++j4) {
                    double tmp;
                    offDiagonal[j4] = tmp = offDiagonal[j4] / diagElement;
                    scaledElementSum += tmp * diagonal[j4];
                }
                double offDiagScalingFactor = scaledElementSum / (diagElement + diagElement);
                for (j2 = 0; j2 < i; ++j2) {
                    int n = j2;
                    offDiagonal[n] = offDiagonal[n] - offDiagScalingFactor * diagonal[j2];
                }
                for (j2 = 0; j2 < i; ++j2) {
                    double tmpDiag = diagonal[j2];
                    double tmpOffDiag = offDiagonal[j2];
                    for (int k4 = j2; k4 < i; ++k4) {
                        double[] dArray = transformValues[k4];
                        int n = j2;
                        dArray[n] = dArray[n] - (tmpDiag * offDiagonal[k4] + tmpOffDiag * diagonal[k4]);
                    }
                    diagonal[j2] = transformValues[i - 1][j2];
                    transformValues[i][j2] = 0.0;
                }
            }
            diagonal[i] = diagElement;
        }
        int dimMinusOne = this.dim1 - 1;
        for (int i = 0; i < dimMinusOne; ++i) {
            transformValues[dimMinusOne][i] = transformValues[i][i];
            transformValues[i][i] = 1.0;
            int nextIdx = i + 1;
            double nextDiag = diagonal[nextIdx];
            if (nextDiag == 0.0) continue;
            for (k = 0; k < nextIdx; ++k) {
                diagonal[k] = transformValues[k][nextIdx] / nextDiag;
            }
            for (j = 0; j < nextIdx; ++j) {
                int k5;
                double scaleAccumulator = 0.0;
                for (k5 = 0; k5 < nextIdx; ++k5) {
                    scaleAccumulator += transformValues[k5][nextIdx] * transformValues[k5][j];
                }
                for (k5 = 0; k5 < nextIdx; ++k5) {
                    double[] dArray = transformValues[k5];
                    int n = j;
                    dArray[n] = dArray[n] - scaleAccumulator * diagonal[k5];
                }
            }
            for (j = 0; j < nextIdx; ++j) {
                transformValues[j][nextIdx] = 0.0;
            }
        }
        for (int j5 = 0; j5 < this.dim1; ++j5) {
            diagonal[j5] = transformValues[dimMinusOne][j5];
            transformValues[dimMinusOne][j5] = 0.0;
        }
        transformValues[dimMinusOne][dimMinusOne] = 1.0;
        offDiagonal[0] = 0.0;
        DenseVector diagVector = DenseVector.createDenseVector(diagonal);
        DenseVector offDiagVector = DenseVector.createDenseVector(offDiagonal);
        DenseMatrix householderMatrix = new DenseMatrix(transform);
        int maxItr = 35;
        double eps = Double.longBitsToDouble(4372995238176751616L);
        System.arraycopy(offDiagonal, 1, offDiagonal, 0, dimMinusOne);
        offDiagonal[dimMinusOne] = 0.0;
        double diagAccum = 0.0;
        double largestDiagSum = 0.0;
        for (int i = 0; i < this.dim1; ++i) {
            int idx;
            largestDiagSum = Math.max(largestDiagSum, Math.abs(diagonal[i]) + Math.abs(offDiagonal[i]));
            double testVal = largestDiagSum * eps;
            for (idx = i; idx < this.dim1 && !(Math.abs(offDiagonal[idx]) <= testVal); ++idx) {
            }
            if (idx > i) {
                int iter = 0;
                do {
                    if (iter > 35) {
                        logger.fine("Exceeded QL iteration count in eigenDecomposition");
                        return Optional.empty();
                    }
                    ++iter;
                    double curDiag = diagonal[i];
                    double shift = (diagonal[i + 1] - curDiag) / (2.0 * offDiagonal[i]);
                    double shiftLength = shift < 0.0 ? -Math.hypot(shift, 1.0) : Math.hypot(shift, 1.0);
                    diagonal[i] = offDiagonal[i] / (shift + shiftLength);
                    diagonal[i + 1] = offDiagonal[i] * (shift + shiftLength);
                    double nextDiag = diagonal[i + 1];
                    double diagShift = curDiag - diagonal[i];
                    int j6 = i + 2;
                    while (j6 < this.dim1) {
                        int n = j6++;
                        diagonal[n] = diagonal[n] - diagShift;
                    }
                    diagAccum += diagShift;
                    double partitionDiag = diagonal[idx];
                    double oldOffDiag = offDiagonal[i + 1];
                    double c = 1.0;
                    double c2 = 1.0;
                    double c3 = 1.0;
                    double s = 0.0;
                    double prevS = 0.0;
                    for (int j7 = idx - 1; j7 >= i; --j7) {
                        c3 = c2;
                        c2 = c;
                        prevS = s;
                        double scaledOffDiag = c * offDiagonal[j7];
                        double scaledDiag = c * partitionDiag;
                        double dist = Math.hypot(partitionDiag, offDiagonal[j7]);
                        offDiagonal[j7 + 1] = s * dist;
                        s = offDiagonal[j7] / dist;
                        c = partitionDiag / dist;
                        partitionDiag = c * diagonal[j7] - s * scaledOffDiag;
                        diagonal[j7 + 1] = scaledDiag + s * (c * scaledOffDiag + s * diagonal[j7]);
                        for (int k6 = 0; k6 < this.dim1; ++k6) {
                            double[] row = transformValues[k6];
                            double tmp = row[j7 + 1];
                            row[j7 + 1] = s * row[j7] + c * tmp;
                            row[j7] = c * row[j7] - s * tmp;
                        }
                    }
                    partitionDiag = -s * prevS * c3 * oldOffDiag * offDiagonal[i] / nextDiag;
                    offDiagonal[i] = s * partitionDiag;
                    diagonal[i] = c * partitionDiag;
                } while (Math.abs(offDiagonal[i]) > testVal);
            }
            int n = i;
            diagonal[n] = diagonal[n] + diagAccum;
            offDiagonal[i] = 0.0;
        }
        int[] indices = SortUtil.argsort((double[])diagonal, (boolean)false);
        double[] eigenValues = new double[this.dim1];
        double[][] eigenVectors = new double[this.dim1][this.dim1];
        for (int i = 0; i < indices.length; ++i) {
            eigenValues[i] = diagonal[indices[i]];
            for (int j8 = 0; j8 < this.dim1; ++j8) {
                eigenVectors[j8][i] = transformValues[j8][indices[i]];
            }
        }
        return Optional.of(new EigenDecomposition(new DenseVector(eigenValues), new DenseMatrix(eigenVectors), diagVector, offDiagVector, householderMatrix));
    }

    public String toString() {
        StringBuilder buffer = new StringBuilder();
        buffer.append("DenseMatrix(dim1=");
        buffer.append(this.dim1);
        buffer.append(",dim2=");
        buffer.append(this.dim2);
        buffer.append(",values=\n");
        for (int i = 0; i < this.dim1; ++i) {
            buffer.append("\trow ");
            buffer.append(i);
            buffer.append(" [");
            for (int j = 0; j < this.dim2; ++j) {
                if (this.values[i][j] < 0.0) {
                    buffer.append(String.format("%.15f", this.values[i][j]));
                } else {
                    buffer.append(String.format(" %.15f", this.values[i][j]));
                }
                buffer.append(",");
            }
            buffer.deleteCharAt(buffer.length() - 1);
            buffer.append("];\n");
        }
        buffer.append(")");
        return buffer.toString();
    }

    public MatrixIterator iterator() {
        return new DenseMatrixIterator(this);
    }

    public void normalizeRows(VectorNormalizer normalizer) {
        for (int i = 0; i < this.dim1; ++i) {
            normalizer.normalizeInPlace(this.values[i]);
        }
    }

    public DenseVector columnSum() {
        double[] columnSum = new double[this.dim2];
        for (int i = 0; i < this.dim1; ++i) {
            for (int j = 0; j < this.dim2; ++j) {
                int n = j;
                columnSum[n] = columnSum[n] + this.get(i, j);
            }
        }
        return new DenseVector(columnSum);
    }

    public DenseMatrix selectColumns(int[] columnIndices) {
        if (columnIndices == null || columnIndices.length == 0) {
            throw new IllegalArgumentException("Invalid column indices.");
        }
        DenseMatrix returnVal = new DenseMatrix(this.dim1, columnIndices.length);
        for (int i = 0; i < this.dim1; ++i) {
            for (int j = 0; j < columnIndices.length; ++j) {
                int curIdx = columnIndices[j];
                if (curIdx < 0 || curIdx >= this.dim2) {
                    throw new IllegalArgumentException("Invalid column index, expected [0, " + this.dim2 + "), found " + curIdx);
                }
                returnVal.values[i][j] = this.get(i, curIdx);
            }
        }
        return returnVal;
    }

    public DenseMatrix selectColumns(List<Integer> columnIndices) {
        if (columnIndices == null || columnIndices.isEmpty()) {
            throw new IllegalArgumentException("Invalid column indices.");
        }
        DenseMatrix returnVal = new DenseMatrix(this.dim1, columnIndices.size());
        for (int i = 0; i < this.dim1; ++i) {
            for (int j = 0; j < columnIndices.size(); ++j) {
                int curIdx = columnIndices.get(j);
                if (curIdx < 0 || curIdx >= this.dim2) {
                    throw new IllegalArgumentException("Invalid column index, expected [0, " + this.dim2 + "), found " + curIdx);
                }
                returnVal.values[i][j] = this.get(i, curIdx);
            }
        }
        return returnVal;
    }

    public static final class CholeskyFactorization
    implements Matrix.Factorization {
        private final DenseMatrix lMatrix;

        CholeskyFactorization(DenseMatrix lMatrix) {
            this.lMatrix = lMatrix;
        }

        public DenseMatrix lMatrix() {
            return this.lMatrix;
        }

        @Override
        public int dim1() {
            return this.lMatrix.dim1;
        }

        @Override
        public int dim2() {
            return this.lMatrix.dim2;
        }

        @Override
        public double determinant() {
            double det = 0.0;
            for (int i = 0; i < this.lMatrix.dim1; ++i) {
                det *= this.lMatrix.values[i][i] * this.lMatrix.values[i][i];
            }
            return det;
        }

        @Override
        public DenseVector solve(SGDVector vector) {
            int j;
            double sum;
            int i;
            if (vector.size() != this.lMatrix.dim1) {
                throw new IllegalArgumentException("Size mismatch, expected " + this.lMatrix.dim1 + ", received " + vector.size());
            }
            double[] vectorArr = vector.toArray();
            double[] output = new double[this.lMatrix.dim1];
            for (i = 0; i < this.lMatrix.dim1; ++i) {
                sum = vectorArr[i];
                for (j = i - 1; j >= 0; --j) {
                    sum -= this.lMatrix.values[i][j] * output[j];
                }
                output[i] = sum / this.lMatrix.values[i][i];
            }
            for (i = this.lMatrix.dim1 - 1; i >= 0; --i) {
                sum = output[i];
                for (j = i + 1; j < this.lMatrix.dim1; ++j) {
                    sum -= this.lMatrix.values[j][i] * output[j];
                }
                output[i] = sum / this.lMatrix.values[i][i];
            }
            return new DenseVector(output);
        }

        @Override
        public DenseMatrix solve(Matrix matrix) {
            int k;
            int j;
            int i;
            if (matrix.getDimension1Size() != this.lMatrix.dim1) {
                throw new IllegalArgumentException("Size mismatch, expected " + this.lMatrix.dim1 + ", received " + matrix.getDimension1Size());
            }
            int outputDim1 = this.lMatrix.dim1;
            int outputDim2 = matrix.getDimension2Size();
            DenseMatrix output = new DenseMatrix(matrix);
            double[][] outputArr = output.values;
            for (i = 0; i < outputDim1; ++i) {
                j = 0;
                while (j < outputDim2) {
                    for (k = 0; k < i; ++k) {
                        double[] dArray = outputArr[i];
                        int n = j;
                        dArray[n] = dArray[n] - outputArr[k][j] * this.lMatrix.values[i][k];
                    }
                    double[] dArray = outputArr[i];
                    int n = j++;
                    dArray[n] = dArray[n] / this.lMatrix.values[i][i];
                }
            }
            for (i = outputDim1 - 1; i >= 0; --i) {
                j = 0;
                while (j < outputDim2) {
                    for (k = i + 1; k < outputDim2; ++k) {
                        double[] dArray = outputArr[i];
                        int n = j;
                        dArray[n] = dArray[n] - outputArr[k][j] * this.lMatrix.values[k][i];
                    }
                    double[] dArray = outputArr[i];
                    int n = j++;
                    dArray[n] = dArray[n] / this.lMatrix.values[i][i];
                }
            }
            return output;
        }
    }

    public static final class LUFactorization
    implements Matrix.Factorization {
        private final DenseMatrix lower;
        private final DenseMatrix upper;
        private final int[] permutationArr;
        private final Matrix permutationMatrix;
        private final boolean oddSwaps;

        LUFactorization(DenseMatrix lower, DenseMatrix upper, int[] permutationArr, boolean oddSwaps) {
            this.lower = lower;
            this.upper = upper;
            this.permutationArr = permutationArr;
            SparseVector[] vecs = new SparseVector[permutationArr.length];
            for (int i = 0; i < vecs.length; ++i) {
                vecs[i] = new SparseVector(lower.dim1, new int[]{permutationArr[i]}, new double[]{1.0});
            }
            this.permutationMatrix = DenseSparseMatrix.createFromSparseVectors(vecs);
            this.oddSwaps = oddSwaps;
        }

        public DenseMatrix lower() {
            return this.lower;
        }

        public DenseMatrix upper() {
            return this.upper;
        }

        public int[] permutationArr() {
            return this.permutationArr;
        }

        public Matrix permutationMatrix() {
            return this.permutationMatrix;
        }

        public boolean oddSwaps() {
            return this.oddSwaps;
        }

        @Override
        public int dim1() {
            return this.permutationArr.length;
        }

        @Override
        public int dim2() {
            return this.permutationArr.length;
        }

        @Override
        public double determinant() {
            double det = 0.0;
            for (int i = 0; i < this.upper.dim1; ++i) {
                det *= this.upper.values[i][i];
            }
            if (this.oddSwaps) {
                return -det;
            }
            return det;
        }

        @Override
        public DenseVector solve(SGDVector vector) {
            int k;
            int i;
            if (vector.size() != this.lower.dim1) {
                throw new IllegalArgumentException("Size mismatch, expected " + this.lower.dim1 + ", received " + vector.size());
            }
            double[] vectorArr = vector.toArray();
            double[] output = new double[vectorArr.length];
            for (i = 0; i < this.permutationArr.length; ++i) {
                output[i] = vectorArr[this.permutationArr[i]];
                for (k = 0; k < i; ++k) {
                    int n = i;
                    output[n] = output[n] - this.lower.values[i][k] * output[k];
                }
            }
            for (i = this.permutationArr.length - 1; i >= 0; --i) {
                for (k = i + 1; k < this.permutationArr.length; ++k) {
                    int n = i;
                    output[n] = output[n] - this.upper.values[i][k] * output[k];
                }
                int n = i;
                output[n] = output[n] / this.upper.values[i][i];
            }
            return new DenseVector(output);
        }

        @Override
        public DenseMatrix solve(Matrix matrix) {
            int k;
            int j;
            int i;
            if (matrix.getDimension1Size() != this.lower.dim1) {
                throw new IllegalArgumentException("Size mismatch, expected " + this.lower.dim1 + ", received " + matrix.getDimension1Size());
            }
            int outputDim1 = this.lower.dim1;
            int outputDim2 = matrix.getDimension2Size();
            double[][] output = new double[this.lower.dim1][];
            for (i = 0; i < outputDim1; ++i) {
                int permutedIdx = this.permutationArr[i];
                for (int j2 = 0; j2 < outputDim2; ++j2) {
                    output[i] = matrix.getRow(permutedIdx).toArray();
                }
            }
            for (i = 0; i < outputDim1; ++i) {
                for (j = i + 1; j < outputDim1; ++j) {
                    for (k = 0; k < outputDim2; ++k) {
                        double[] dArray = output[j];
                        int n = k;
                        dArray[n] = dArray[n] - output[i][k] * this.lower.values[j][i];
                    }
                }
            }
            for (i = outputDim1 - 1; i >= 0; --i) {
                j = 0;
                while (j < outputDim2) {
                    double[] dArray = output[i];
                    int n = j++;
                    dArray[n] = dArray[n] / this.upper.values[i][i];
                }
                for (j = 0; j < i; ++j) {
                    for (k = 0; k < outputDim2; ++k) {
                        double[] dArray = output[j];
                        int n = k;
                        dArray[n] = dArray[n] - output[i][k] * this.upper.values[j][i];
                    }
                }
            }
            return new DenseMatrix(output);
        }
    }

    public static final class EigenDecomposition
    implements Matrix.Factorization {
        private final DenseVector eigenvalues;
        private final DenseMatrix eigenvectors;
        private final DenseVector diagonal;
        private final DenseVector offDiagonal;
        private final DenseMatrix householderMatrix;

        EigenDecomposition(DenseVector eigenvalues, DenseMatrix eigenvectors, DenseVector diagonal, DenseVector offDiagonal, DenseMatrix householderMatrix) {
            this.eigenvalues = eigenvalues;
            this.eigenvectors = eigenvectors;
            this.diagonal = diagonal;
            this.offDiagonal = offDiagonal;
            this.householderMatrix = householderMatrix;
        }

        public DenseVector eigenvalues() {
            return this.eigenvalues;
        }

        public DenseMatrix eigenvectors() {
            return this.eigenvectors;
        }

        public DenseVector diagonal() {
            return this.diagonal;
        }

        public DenseVector offDiagonal() {
            return this.offDiagonal;
        }

        public DenseMatrix householderMatrix() {
            return this.householderMatrix;
        }

        @Override
        public int dim1() {
            return this.eigenvalues.size();
        }

        @Override
        public int dim2() {
            return this.eigenvalues.size();
        }

        @Override
        public double determinant() {
            return this.eigenvalues.reduce(1.0, DoubleUnaryOperator.identity(), (a, b) -> a * b);
        }

        public boolean positiveEigenvalues() {
            return this.eigenvalues.reduce(true, DoubleUnaryOperator.identity(), (value, bool) -> bool != false && value > 0.0);
        }

        public boolean nonSingular() {
            return this.eigenvalues.reduce(true, DoubleUnaryOperator.identity(), (value, bool) -> bool != false && value != 0.0);
        }

        public DenseVector getEigenVector(int i) {
            if (i < 0 || i > this.eigenvectors.dim1) {
                throw new IllegalArgumentException("Invalid index, must be [0," + this.eigenvectors.dim1 + "), found " + i);
            }
            return this.eigenvectors.getColumn(i);
        }

        @Override
        public DenseVector solve(SGDVector vector) {
            if (vector.size() != this.eigenvectors.dim1) {
                throw new IllegalArgumentException("Size mismatch, expected " + this.eigenvectors.dim1 + ", received " + vector.size());
            }
            double[] output = new double[vector.size()];
            for (int i = 0; i < output.length; ++i) {
                DenseVector eigenVector = this.getEigenVector(i);
                double value = vector.dot(eigenVector) / this.eigenvalues.get(i);
                for (int j = 0; j < output.length; ++j) {
                    int n = j;
                    output[n] = output[n] + value * eigenVector.get(j);
                }
            }
            return new DenseVector(output);
        }

        @Override
        public DenseMatrix solve(Matrix matrix) {
            if (matrix.getDimension1Size() != this.eigenvectors.dim1) {
                throw new IllegalArgumentException("Size mismatch, expected " + this.eigenvectors.dim1 + ", received " + matrix.getDimension1Size());
            }
            int outputDim1 = this.eigenvalues.size();
            int outputDim2 = matrix.getDimension2Size();
            double[][] output = new double[outputDim1][outputDim2];
            for (int k = 0; k < outputDim2; ++k) {
                SGDVector column = matrix.getColumn(k);
                for (int i = 0; i < outputDim1; ++i) {
                    DenseVector eigen = this.getEigenVector(i);
                    double value = eigen.dot(column) / this.eigenvalues.get(i);
                    for (int j = 0; j < output.length; ++j) {
                        double[] dArray = output[j];
                        int n = k;
                        dArray[n] = dArray[n] + value * eigen.get(j);
                    }
                }
            }
            return new DenseMatrix(output);
        }
    }

    private class DenseMatrixIterator
    implements MatrixIterator {
        private final DenseMatrix matrix;
        private final MatrixTuple tuple;
        private int i;
        private int j;

        DenseMatrixIterator(DenseMatrix matrix) {
            this.matrix = matrix;
            this.tuple = new MatrixTuple();
            this.i = 0;
            this.j = 0;
        }

        @Override
        public MatrixTuple getReference() {
            return this.tuple;
        }

        @Override
        public boolean hasNext() {
            return this.i < this.matrix.dim1 && this.j < this.matrix.dim2;
        }

        @Override
        public MatrixTuple next() {
            if (!this.hasNext()) {
                throw new NoSuchElementException("Off the end of the iterator.");
            }
            this.tuple.i = this.i;
            this.tuple.j = this.j;
            this.tuple.value = this.matrix.values[this.i][this.j];
            if (this.j < DenseMatrix.this.dim2 - 1) {
                ++this.j;
            } else {
                ++this.i;
                this.j = 0;
            }
            return this.tuple;
        }
    }
}

