/*
 * Decompiled with CFR 0.152.
 */
package io.lucenia.security.util.ratetracking;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.RemovalListener;
import com.google.common.cache.RemovalNotification;
import io.lucenia.security.util.ratetracking.RateTracker;
import java.util.Arrays;
import java.util.Optional;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.function.LongSupplier;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

public class HeapBasedRateTracker<ClientIdType>
implements RateTracker<ClientIdType> {
    private final Logger log = LogManager.getLogger(this.getClass());
    private final Cache<ClientIdType, ClientRecord> cache;
    private final LongSupplier timeProvider;
    private final long timeWindowMs;
    private final int maxTimeOffsets;

    public HeapBasedRateTracker(long timeWindowMs, int allowedTries, int maxEntries) {
        this(timeWindowMs, allowedTries, maxEntries, null);
    }

    public HeapBasedRateTracker(long timeWindowMs, int allowedTries, int maxEntries, LongSupplier timeProvider) {
        if (allowedTries < 2) {
            throw new IllegalArgumentException("allowedTries must be >= 2");
        }
        this.timeWindowMs = timeWindowMs;
        this.maxTimeOffsets = allowedTries > 2 ? allowedTries - 2 : 0;
        this.timeProvider = Optional.ofNullable(timeProvider).orElse(System::currentTimeMillis);
        this.cache = CacheBuilder.newBuilder().expireAfterAccess(this.timeWindowMs, TimeUnit.MILLISECONDS).maximumSize((long)maxEntries).concurrencyLevel(4).removalListener(new RemovalListener<ClientIdType, ClientRecord>(){

            public void onRemoval(RemovalNotification<ClientIdType, ClientRecord> notification) {
                if (HeapBasedRateTracker.this.log.isDebugEnabled()) {
                    HeapBasedRateTracker.this.log.debug("Removing {}", notification.getKey());
                }
            }
        }).build();
    }

    @Override
    public boolean track(ClientIdType clientId) {
        try {
            ClientRecord clientRecord = (ClientRecord)this.cache.get(clientId, () -> new ClientRecord());
            boolean result = clientRecord.track();
            if (this.log.isDebugEnabled()) {
                this.log.debug("track({}): {} => {}", clientId, (Object)clientRecord, (Object)result);
            }
            return result;
        }
        catch (ExecutionException e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    public void reset(ClientIdType clientId) {
        this.cache.invalidate(clientId);
    }

    private class ClientRecord {
        private long startTime = -1L;
        private final int[] timeOffsets;
        private short timeOffsetStart;
        private short timeOffsetEnd;

        private ClientRecord() {
            this.timeOffsets = new int[HeapBasedRateTracker.this.maxTimeOffsets];
            this.timeOffsetStart = (short)-1;
            this.timeOffsetEnd = (short)-1;
        }

        synchronized boolean track() {
            long timestamp = HeapBasedRateTracker.this.timeProvider.getAsLong();
            if (this.startTime == -1L || timestamp - this.getMostRecent() >= HeapBasedRateTracker.this.timeWindowMs) {
                this.startTime = timestamp;
                this.timeOffsetEnd = (short)-1;
                this.timeOffsetStart = (short)-1;
                return false;
            }
            if (timestamp - this.startTime >= HeapBasedRateTracker.this.timeWindowMs) {
                this.removeExpiredEntries(timestamp);
            } else if (this.isFull()) {
                this.shiftFull(timestamp);
                return true;
            }
            if (this.startTime == -1L) {
                this.startTime = timestamp;
                this.timeOffsetEnd = (short)-1;
                this.timeOffsetStart = (short)-1;
                return false;
            }
            if (this.timeOffsetStart == -1) {
                this.timeOffsets[0] = (int)(timestamp - this.startTime);
                this.timeOffsetStart = 0;
                this.timeOffsetEnd = 0;
            } else {
                short newEnd = this.next(this.timeOffsetEnd);
                this.timeOffsets[newEnd] = (int)(timestamp - this.startTime);
                this.timeOffsetEnd = newEnd;
            }
            return false;
        }

        private boolean isFull() {
            return this.startTime != 0L && (this.timeOffsetStart == this.timeOffsetEnd + 1 || this.timeOffsetStart == 0 && this.timeOffsetEnd == this.timeOffsets.length - 1 || this.timeOffsets.length == 0);
        }

        private void shiftFull(long timestamp) {
            short second;
            if (this.timeOffsets.length == 0) {
                this.startTime = timestamp;
                return;
            }
            int shiftOffset = this.timeOffsets[this.timeOffsetStart];
            this.startTime += (long)shiftOffset;
            short oldStart = this.timeOffsetStart;
            short i = second = this.next(this.timeOffsetStart);
            while (true) {
                short s = i;
                this.timeOffsets[s] = this.timeOffsets[s] - shiftOffset;
                if (i == this.timeOffsetEnd) break;
                if ((i = (short)(i + 1)) < this.timeOffsets.length) continue;
                i = 0;
            }
            this.timeOffsetStart = second;
            this.timeOffsets[oldStart] = (int)(timestamp - this.startTime);
            this.timeOffsetEnd = oldStart;
        }

        private long getMostRecent() {
            if (this.timeOffsetStart == -1) {
                return this.startTime;
            }
            return this.startTime + (long)this.timeOffsets[this.timeOffsetEnd];
        }

        private void removeExpiredEntries(long timestamp) {
            short firstNonExpired = this.findFirstNonExpiredEntry(timestamp);
            if (firstNonExpired == -1) {
                this.startTime = -1L;
                this.timeOffsetEnd = (short)-1;
                this.timeOffsetStart = (short)-1;
                return;
            }
            long newStartTime = this.startTime + (long)this.timeOffsets[firstNonExpired];
            if (firstNonExpired == this.timeOffsetEnd) {
                this.startTime = newStartTime;
                this.timeOffsetEnd = (short)-1;
                this.timeOffsetStart = (short)-1;
                return;
            }
            short secondNonExpired = this.next(firstNonExpired);
            int offsetBetweenOldAndNew = this.timeOffsets[firstNonExpired];
            short i = secondNonExpired;
            while (true) {
                short s = i;
                this.timeOffsets[s] = this.timeOffsets[s] - offsetBetweenOldAndNew;
                if (i == this.timeOffsetEnd) break;
                if ((i = (short)(i + 1)) < this.timeOffsets.length) continue;
                i = 0;
            }
            this.startTime = newStartTime;
            this.timeOffsetStart = secondNonExpired;
        }

        private short next(short i) {
            if ((i = (short)(i + 1)) >= this.timeOffsets.length) {
                i = 0;
            }
            return i;
        }

        private short findFirstNonExpiredEntry(long timestamp) {
            short i = this.timeOffsetStart;
            if (i == -1) {
                return -1;
            }
            while (true) {
                if (timestamp - (this.startTime + (long)this.timeOffsets[i]) < HeapBasedRateTracker.this.timeWindowMs) {
                    return i;
                }
                if (i == this.timeOffsetEnd) break;
                if ((i = (short)(i + 1)) < this.timeOffsets.length) continue;
                i = 0;
            }
            return -1;
        }

        public String toString() {
            return "ClientRecord [startTime=" + this.startTime + ", timeOffsets=" + Arrays.toString(this.timeOffsets) + ", timeOffsetStart=" + this.timeOffsetStart + ", timeOffsetEnd=" + this.timeOffsetEnd + "]";
        }
    }
}

