/*
 * Decompiled with CFR 0.152.
 */
package io.lucenia.indexmanagement.rollup;

import io.lucenia.indexmanagement.rollup.model.Rollup;
import io.lucenia.indexmanagement.rollup.model.RollupMetadata;
import io.lucenia.indexmanagement.rollup.settings.RollupSettings;
import io.lucenia.indexmanagement.rollup.util.RollupUtils;
import io.skylite.SkyliteExceptionsHelper;
import io.skylite.SkyliteSecurityException;
import io.skylite.common.action.ActionListener;
import io.skylite.common.unit.TimeValue;
import io.skylite.core.action.bulk.BackoffPolicy;
import io.skylite.core.action.search.SearchPhaseExecutionException;
import io.skylite.core.action.search.SearchResponse;
import io.skylite.core.action.search.TransportSearchAction;
import io.skylite.core.aggregations.MultiBucketConsumerService;
import io.skylite.core.client.Client;
import io.skylite.core.cluster.service.ClusterService;
import io.skylite.core.common.breaker.CircuitBreakingException;
import io.skylite.core.search.SearchRequest;
import io.skylite.core.settings.Settings;
import io.skylite.core.transport.RemoteTransportException;
import java.time.Instant;
import java.util.Arrays;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

public class RollupSearchService {
    private final Logger logger = LogManager.getLogger(RollupSearchService.class);
    private final Client client;
    private volatile BackoffPolicy retrySearchPolicy;
    private volatile int retryCount;
    private volatile long retryDelayMillis;
    private volatile TimeValue cancelAfterTimeInterval;

    public RollupSearchService(Settings settings, ClusterService clusterService, Client client) {
        this.client = client;
        int count = (Integer)RollupSettings.ROLLUP_SEARCH_BACKOFF_COUNT.get(settings);
        long delayMillis = ((TimeValue)RollupSettings.ROLLUP_SEARCH_BACKOFF_MILLIS.get(settings)).millis();
        this.retrySearchPolicy = BackoffPolicy.constantBackoff((TimeValue)((TimeValue)RollupSettings.ROLLUP_SEARCH_BACKOFF_MILLIS.get(settings)), (int)count);
        this.retryCount = count;
        this.retryDelayMillis = delayMillis;
        this.cancelAfterTimeInterval = (TimeValue)TransportSearchAction.SEARCH_CANCEL_AFTER_TIME_INTERVAL_SETTING.get(settings);
        clusterService.getClusterSettings().addSettingsUpdateConsumer(RollupSettings.ROLLUP_SEARCH_BACKOFF_MILLIS, RollupSettings.ROLLUP_SEARCH_BACKOFF_COUNT, (millis, count2) -> {
            this.retrySearchPolicy = BackoffPolicy.constantBackoff((TimeValue)millis, (int)count2);
            this.retryCount = count2;
            this.retryDelayMillis = millis.millis();
        });
        clusterService.getClusterSettings().addSettingsUpdateConsumer(TransportSearchAction.SEARCH_CANCEL_AFTER_TIME_INTERVAL_SETTING, it -> {
            this.cancelAfterTimeInterval = it;
        });
    }

    public boolean shouldProcessRollup(Rollup rollup, RollupMetadata metadata) {
        if (!rollup.getEnabled()) {
            return false;
        }
        if (metadata == null) {
            return true;
        }
        if (metadata.getStatus() == RollupMetadata.Status.RETRY) {
            return true;
        }
        if (Arrays.asList(RollupMetadata.Status.STOPPED, RollupMetadata.Status.FAILED).contains((Object)metadata.getStatus())) {
            return false;
        }
        if (metadata.getAfterKey() != null) {
            return true;
        }
        if (!rollup.getContinuous()) {
            if (Arrays.asList(RollupMetadata.Status.INIT, RollupMetadata.Status.STARTED).contains((Object)metadata.getStatus())) {
                return true;
            }
            this.logger.debug("Non-continuous job [{}] is not processing next window [{}]", (Object)rollup.getId(), (Object)metadata);
            return false;
        }
        return this.hasNextFullWindow(rollup, metadata);
    }

    private boolean hasNextFullWindow(Rollup rollup, RollupMetadata metadata) {
        long delay = rollup.getDelay() != null ? rollup.getDelay() : 0L;
        return Instant.now().isAfter(metadata.getContinuous().getNextWindowEndTime().plusMillis(delay));
    }

    public void executeCompositeSearch(Rollup job, RollupMetadata metadata, ActionListener<RollupSearchResult> listener) {
        this.executeSearchWithRetry(job, metadata, 0, listener);
    }

    private void executeSearchWithRetry(Rollup job, RollupMetadata metadata, int retryAttempt, ActionListener<RollupSearchResult> listener) {
        try {
            float decay = (float)Math.pow(2.0, retryAttempt);
            int pageSize = Math.max(1, (int)((float)job.getPageSize() / decay));
            if (retryAttempt > 0) {
                this.logger.warn("Composite search failed for rollup, retrying [#{}] - reducing page size from {} to {}", (Object)retryAttempt, (Object)job.getPageSize(), (Object)pageSize);
            }
            SearchRequest searchRequest = RollupUtils.getRollupSearchRequest(new Rollup.Builder(job).pageSize(pageSize).build(), metadata);
            TimeValue cancelTimeoutTimeValue = TimeValue.timeValueMinutes((long)this.getCancelAfterTimeInterval(this.cancelAfterTimeInterval.minutes()));
            searchRequest.setCancelAfterTimeInterval(cancelTimeoutTimeValue);
            this.client.search(searchRequest, ActionListener.wrap(response -> listener.onResponse((Object)new RollupSearchResult.Success((SearchResponse)response)), e -> this.handleSearchError(job, metadata, retryAttempt, (Throwable)e, listener)));
        }
        catch (Exception e2) {
            this.handleSearchError(job, metadata, retryAttempt, e2, listener);
        }
    }

    private void handleSearchError(Rollup job, RollupMetadata metadata, int currentAttempt, Throwable e, ActionListener<RollupSearchResult> listener) {
        Exception unwrappedException;
        this.logger.error(e.getMessage(), e.getCause());
        if (e instanceof SearchPhaseExecutionException) {
            SearchPhaseExecutionException spee = (SearchPhaseExecutionException)e;
            unwrappedException = spee.shardFailures().length == 0 ? (Exception)SkyliteExceptionsHelper.unwrapCause((Throwable)e) : (Exception)SkyliteExceptionsHelper.unwrapCause((Throwable)spee.shardFailures()[0].getCause());
        } else if (e instanceof RemoteTransportException) {
            unwrappedException = (Exception)SkyliteExceptionsHelper.unwrapCause((Throwable)((RemoteTransportException)e));
        } else if (e instanceof CircuitBreakingException || e instanceof MultiBucketConsumerService.TooManyBucketsException) {
            unwrappedException = (Exception)e;
        } else {
            if (e instanceof SkyliteSecurityException) {
                listener.onResponse((Object)new RollupSearchResult.Failure("Cannot search data in source index/s - missing required index permissions: " + e.getLocalizedMessage(), (Exception)e));
                return;
            }
            unwrappedException = (Exception)e;
        }
        int nextAttempt = currentAttempt + 1;
        if (this.shouldRetry(unwrappedException) && nextAttempt <= this.retryCount) {
            long delayMillis;
            long l = delayMillis = currentAttempt > 0 ? this.retryDelayMillis : 0L;
            if (delayMillis > 0L) {
                try {
                    Thread.sleep(delayMillis);
                }
                catch (InterruptedException ie) {
                    Thread.currentThread().interrupt();
                    listener.onResponse((Object)new RollupSearchResult.Failure(ie));
                    return;
                }
            }
            this.executeSearchWithRetry(job, metadata, nextAttempt, listener);
        } else {
            listener.onResponse((Object)new RollupSearchResult.Failure(unwrappedException));
        }
    }

    private boolean shouldRetry(Exception e) {
        return e instanceof CircuitBreakingException || e instanceof MultiBucketConsumerService.TooManyBucketsException;
    }

    private long getCancelAfterTimeInterval(long givenInterval) {
        if (givenInterval == -1L) {
            return givenInterval;
        }
        return Math.max(this.cancelAfterTimeInterval.minutes(), 10L);
    }

    public RollupSearchResult executeCompositeSearchSync(Rollup job, RollupMetadata metadata) throws Exception {
        final CompletableFuture future = new CompletableFuture();
        this.executeCompositeSearch(job, metadata, new ActionListener<RollupSearchResult>(){

            public void onResponse(RollupSearchResult result) {
                future.complete(result);
            }

            public void onFailure(Exception e) {
                future.completeExceptionally(e);
            }
        });
        try {
            return (RollupSearchResult)future.get();
        }
        catch (ExecutionException e) {
            throw (Exception)e.getCause();
        }
    }

    public static abstract class RollupSearchResult {

        public static class Failure
        extends RollupSearchResult {
            private final String message;
            private final Exception cause;

            public Failure(Exception cause) {
                this("An error occurred while searching the rollup source index", cause);
            }

            public Failure(String message, Exception cause) {
                this.message = message;
                this.cause = cause;
            }

            public String getMessage() {
                return this.message;
            }

            public Exception getCause() {
                return this.cause;
            }
        }

        public static class Success
        extends RollupSearchResult {
            private final SearchResponse searchResponse;

            public Success(SearchResponse searchResponse) {
                this.searchResponse = searchResponse;
            }

            public SearchResponse getSearchResponse() {
                return this.searchResponse;
            }
        }
    }
}

