/*
 * Decompiled with CFR 0.152.
 */
package io.lucenia.indexmanagement.transform;

import io.lucenia.indexmanagement.transform.exceptions.TransformValidationException;
import io.lucenia.indexmanagement.transform.model.Transform;
import io.lucenia.indexmanagement.transform.model.TransformValidationResult;
import io.lucenia.indexmanagement.transform.settings.TransformSettings;
import io.skylite.SkyliteExceptionsHelper;
import io.skylite.SkyliteSecurityException;
import io.skylite.common.action.ActionListener;
import io.skylite.core.action.ActionRequest;
import io.skylite.core.action.ActionType;
import io.skylite.core.action.admin.cluster.health.ClusterHealthAction;
import io.skylite.core.action.admin.cluster.health.ClusterHealthRequest;
import io.skylite.core.action.admin.cluster.health.ClusterHealthResponse;
import io.skylite.core.action.admin.indices.mapping.get.GetMappingsRequest;
import io.skylite.core.action.admin.indices.mapping.get.GetMappingsResponse;
import io.skylite.core.action.support.IndicesOptions;
import io.skylite.core.client.Client;
import io.skylite.core.cluster.metadata.IndexNameExpressionResolver;
import io.skylite.core.cluster.metadata.MappingMetadata;
import io.skylite.core.cluster.service.ClusterService;
import io.skylite.core.monitor.jvm.JvmService;
import io.skylite.core.settings.Settings;
import io.skylite.core.transport.RemoteTransportException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

public class TransformValidator {
    private static final Logger logger = LogManager.getLogger(TransformValidator.class);
    private final IndexNameExpressionResolver indexNameExpressionResolver;
    private final ClusterService clusterService;
    private final Client client;
    private final Settings settings;
    private final JvmService jvmService;
    private volatile boolean circuitBreakerEnabled;
    private volatile int circuitBreakerJvmThreshold;

    public TransformValidator(IndexNameExpressionResolver indexNameExpressionResolver, ClusterService clusterService, Client client, Settings settings, JvmService jvmService) {
        this.indexNameExpressionResolver = indexNameExpressionResolver;
        this.clusterService = clusterService;
        this.client = client;
        this.settings = settings;
        this.jvmService = jvmService;
        this.circuitBreakerEnabled = (Boolean)TransformSettings.TRANSFORM_CIRCUIT_BREAKER_ENABLED.get(settings);
        this.circuitBreakerJvmThreshold = (Integer)TransformSettings.TRANSFORM_CIRCUIT_BREAKER_JVM_THRESHOLD.get(settings);
        clusterService.getClusterSettings().addSettingsUpdateConsumer(TransformSettings.TRANSFORM_CIRCUIT_BREAKER_ENABLED, value -> {
            this.circuitBreakerEnabled = value;
        });
        clusterService.getClusterSettings().addSettingsUpdateConsumer(TransformSettings.TRANSFORM_CIRCUIT_BREAKER_JVM_THRESHOLD, value -> {
            this.circuitBreakerJvmThreshold = value;
        });
    }

    public TransformValidationResult validate(final Transform transform) throws TransformValidationException {
        String errorMessage = "Failed to validate the transform job";
        logger.debug("Starting validation for transform [{}]", (Object)transform.getId());
        try {
            ArrayList<String> issues = new ArrayList<String>();
            if (this.circuitBreakerEnabled && this.jvmService.stats().getMem().getHeapUsedPercent() > this.circuitBreakerJvmThreshold) {
                logger.warn("Circuit breaker triggered for transform [{}], JVM usage above threshold [{}]", (Object)transform.getId(), (Object)this.circuitBreakerJvmThreshold);
                issues.add("The cluster is breaching the jvm usage threshold [" + this.circuitBreakerJvmThreshold + "], cannot execute the transform");
                return new TransformValidationResult(issues.isEmpty(), issues);
            }
            logger.debug("Resolving concrete indices for transform [{}], sourceIndex=[{}]", (Object)transform.getId(), (Object)transform.getSourceIndex());
            Object[] concreteIndices = this.indexNameExpressionResolver.concreteIndexNames(this.clusterService.state(), IndicesOptions.lenientExpand(), true, new String[]{transform.getSourceIndex()});
            logger.debug("Resolved [{}] concrete indices for transform [{}]: {}", (Object)concreteIndices.length, (Object)transform.getId(), (Object)Arrays.toString(concreteIndices));
            if (concreteIndices.length == 0) {
                logger.warn("No source indices found for transform [{}]", (Object)transform.getId());
                return new TransformValidationResult(false, Arrays.asList("No specified source index exist in the cluster"));
            }
            logger.debug("Checking cluster health for indices {} for transform [{}]", (Object)Arrays.toString(concreteIndices), (Object)transform.getId());
            ClusterHealthRequest request = new ClusterHealthRequest().indices((String[])concreteIndices).waitForYellowStatus();
            final CompletableFuture future = new CompletableFuture();
            this.client.execute((ActionType)ClusterHealthAction.INSTANCE, (ActionRequest)request, (ActionListener)new ActionListener<ClusterHealthResponse>(){

                public void onResponse(ClusterHealthResponse response) {
                    logger.debug("Cluster health response for transform [{}]: status={}, timedOut={}", (Object)transform.getId(), (Object)response.getStatus(), (Object)response.isTimedOut());
                    future.complete(response);
                }

                public void onFailure(Exception e) {
                    logger.error("Failed to get cluster health for transform [{}]: {} {}", (Object)transform.getId(), (Object)e.getMessage(), (Object)e);
                    future.completeExceptionally(e);
                }
            });
            ClusterHealthResponse response = (ClusterHealthResponse)future.get();
            if (response.isTimedOut()) {
                logger.warn("Cluster health check timed out for transform [{}]", (Object)transform.getId());
                issues.add("Cannot determine that the requested source indices are healthy");
                return new TransformValidationResult(issues.isEmpty(), issues);
            }
            logger.debug("Validating indices for transform [{}]", (Object)transform.getId());
            for (Object index : concreteIndices) {
                logger.debug("Validating index [{}] for transform [{}]", index, (Object)transform.getId());
                issues.addAll(this.validateIndex((String)index, transform));
            }
            logger.debug("Validation complete for transform [{}], valid={}, issues={}", (Object)transform.getId(), (Object)issues.isEmpty(), issues);
            return new TransformValidationResult(issues.isEmpty(), issues);
        }
        catch (RemoteTransportException e) {
            logger.error("RemoteTransportException during validation of transform [{}]: {} {}", (Object)transform.getId(), (Object)e.getMessage(), (Object)e);
            Exception unwrappedException = (Exception)SkyliteExceptionsHelper.unwrapCause((Throwable)e);
            throw new TransformValidationException(errorMessage, unwrappedException);
        }
        catch (SkyliteSecurityException e) {
            logger.error("SecurityException during validation of transform [{}]: {} {}", (Object)transform.getId(), (Object)e.getMessage(), (Object)e);
            throw new TransformValidationException(errorMessage + " - missing required index permissions: " + e.getLocalizedMessage());
        }
        catch (ExecutionException e) {
            Throwable cause = e.getCause();
            logger.error("ExecutionException during validation of transform [{}], cause={}: {} {}", (Object)transform.getId(), (Object)(cause != null ? cause.getClass().getName() : "null"), (Object)(cause != null ? cause.getMessage() : e.getMessage()), (Object)e);
            if (cause instanceof TransformValidationException) {
                throw (TransformValidationException)cause;
            }
            throw new TransformValidationException(errorMessage, e);
        }
        catch (Exception e) {
            logger.error("Exception during validation of transform [{}]: {} {}", (Object)transform.getId(), (Object)e.getMessage(), (Object)e);
            throw new TransformValidationException(errorMessage, e);
        }
    }

    private List<String> validateIndex(final String index, final Transform transform) throws ExecutionException, InterruptedException {
        logger.debug("Getting mappings for index [{}] for transform [{}]", (Object)index, (Object)transform.getId());
        GetMappingsRequest request = (GetMappingsRequest)new GetMappingsRequest().indices(new String[]{index});
        final CompletableFuture future = new CompletableFuture();
        this.client.admin().indices().getMappings(request, (ActionListener)new ActionListener<GetMappingsResponse>(){

            public void onResponse(GetMappingsResponse response) {
                logger.debug("Got mappings response for index [{}] for transform [{}]", (Object)index, (Object)transform.getId());
                future.complete(response);
            }

            public void onFailure(Exception e) {
                logger.error("Failed to get mappings for index [{}] for transform [{}]: {} {}", (Object)index, (Object)transform.getId(), (Object)e.getMessage(), (Object)e);
                future.completeExceptionally(e);
            }
        });
        GetMappingsResponse result = (GetMappingsResponse)future.get();
        if (result == null) {
            logger.error("GetMappingResponse for index [{}] was null for transform [{}]", (Object)index, (Object)transform.getId());
            throw new IllegalStateException("GetMappingResponse for [" + index + "] was null");
        }
        List<String> issues = TransformValidator.validateMappingsResponse(index, result, transform);
        logger.debug("Validated mappings for index [{}] for transform [{}], issues={}", (Object)index, (Object)transform.getId(), issues);
        return issues;
    }

    public static List<String> validateMappingsResponse(String index, GetMappingsResponse response, Transform transform) {
        ArrayList<String> issues = new ArrayList<String>();
        MappingMetadata indexTypeMappings = (MappingMetadata)response.mappings().get(index);
        if (indexTypeMappings == null) {
            issues.add("Source index [" + index + "] mappings are empty, cannot validate the job.");
            return issues;
        }
        Map indexMappingSource = indexTypeMappings.sourceAsMap();
        transform.getGroups().forEach(group -> {
            if (!group.canBeRealizedInMappings(indexMappingSource)) {
                issues.add("Cannot find field [" + group.getSourceField() + "] that can be grouped as [" + group.getType().getType() + "] in [" + index + "].");
            }
        });
        return issues;
    }

    public Settings getSettings() {
        return this.settings;
    }
}

