/*
 * Decompiled with CFR 0.152.
 */
package org.opensearch.cluster.coordination;

import io.skylite.SkyliteException;
import io.skylite.common.Nullable;
import io.skylite.common.lease.Releasable;
import io.skylite.common.transport.TransportRequestOptions;
import io.skylite.common.unit.TimeValue;
import io.skylite.core.cluster.node.DiscoveryNode;
import io.skylite.core.cluster.node.DiscoveryNodes;
import io.skylite.core.common.io.stream.StreamInput;
import io.skylite.core.common.io.stream.StreamOutput;
import io.skylite.core.settings.Setting;
import io.skylite.core.settings.Settings;
import io.skylite.core.settings.spi.SettingsProvider;
import io.skylite.core.transport.ConnectTransportException;
import io.skylite.core.transport.NodeDisconnectedException;
import io.skylite.core.transport.TransportConnection;
import io.skylite.core.transport.TransportConnectionListener;
import io.skylite.core.transport.TransportException;
import io.skylite.core.transport.TransportRequest;
import io.skylite.core.transport.TransportResponse;
import io.skylite.core.transport.TransportResponseHandler;
import io.skylite.core.transport.TransportService;
import java.io.IOException;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Consumer;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.Message;
import org.apache.logging.log4j.message.ParameterizedMessage;
import org.opensearch.cluster.coordination.CoordinationStateRejectedException;
import org.opensearch.cluster.coordination.NodeHealthCheckFailureException;
import org.opensearch.monitor.NodeHealthService;
import org.opensearch.monitor.StatusInfo;

public class LeaderChecker {
    private static final Logger logger = LogManager.getLogger(LeaderChecker.class);
    static final String LEADER_CHECK_ACTION_NAME = "internal:coordination/fault_detection/leader_check";
    public static final Setting<TimeValue> LEADER_CHECK_INTERVAL_SETTING = Setting.timeSetting((String)"cluster.fault_detection.leader_check.interval", (TimeValue)TimeValue.timeValueMillis((long)1000L), (TimeValue)TimeValue.timeValueMillis((long)100L), (Setting.Property[])new Setting.Property[]{Setting.Property.NodeScope});
    public static final Setting<TimeValue> LEADER_CHECK_TIMEOUT_SETTING = Setting.timeSetting((String)"cluster.fault_detection.leader_check.timeout", (TimeValue)TimeValue.timeValueMillis((long)10000L), (TimeValue)TimeValue.timeValueMillis((long)1L), (Setting.Property[])new Setting.Property[]{Setting.Property.NodeScope});
    public static final Setting<Integer> LEADER_CHECK_RETRY_COUNT_SETTING = Setting.intSetting((String)"cluster.fault_detection.leader_check.retry_count", (int)3, (int)1, (Setting.Property[])new Setting.Property[]{Setting.Property.NodeScope});
    private final Settings settings;
    private final TimeValue leaderCheckInterval;
    private final TimeValue leaderCheckTimeout;
    private final int leaderCheckRetryCount;
    private final TransportService transportService;
    private final Consumer<Exception> onLeaderFailure;
    private final NodeHealthService nodeHealthService;
    private AtomicReference<CheckScheduler> currentChecker = new AtomicReference();
    private volatile DiscoveryNodes discoveryNodes;

    LeaderChecker(Settings settings, TransportService transportService, Consumer<Exception> onLeaderFailure, NodeHealthService nodeHealthService) {
        this.settings = settings;
        this.leaderCheckInterval = (TimeValue)LEADER_CHECK_INTERVAL_SETTING.get(settings);
        this.leaderCheckTimeout = (TimeValue)LEADER_CHECK_TIMEOUT_SETTING.get(settings);
        this.leaderCheckRetryCount = (Integer)LEADER_CHECK_RETRY_COUNT_SETTING.get(settings);
        this.transportService = transportService;
        this.onLeaderFailure = onLeaderFailure;
        this.nodeHealthService = nodeHealthService;
        transportService.registerRequestHandler(LEADER_CHECK_ACTION_NAME, "same", false, false, LeaderCheckRequest::new, (request, channel, task) -> {
            this.handleLeaderCheck((LeaderCheckRequest)request);
            channel.sendResponse((TransportResponse)TransportResponse.Empty.INSTANCE);
        });
        transportService.addConnectionListener(new TransportConnectionListener(){

            public void onNodeDisconnected(DiscoveryNode node, TransportConnection connection) {
                LeaderChecker.this.handleDisconnectedNode(node);
            }
        });
    }

    public DiscoveryNode leader() {
        CheckScheduler checkScheduler = this.currentChecker.get();
        return checkScheduler == null ? null : checkScheduler.leader;
    }

    void updateLeader(@Nullable DiscoveryNode leader) {
        assert (!this.transportService.getLocalNode().equals((Object)leader));
        CheckScheduler checkScheduler = leader != null ? new CheckScheduler(leader) : null;
        CheckScheduler previousChecker = this.currentChecker.getAndSet(checkScheduler);
        if (previousChecker != null) {
            previousChecker.close();
        }
        if (checkScheduler != null) {
            checkScheduler.handleWakeUp();
        }
    }

    void setCurrentNodes(DiscoveryNodes discoveryNodes) {
        logger.trace("setCurrentNodes: {}", (Object)discoveryNodes);
        this.discoveryNodes = discoveryNodes;
    }

    boolean currentNodeIsClusterManager() {
        return this.discoveryNodes.isLocalNodeElectedClusterManager();
    }

    private void handleLeaderCheck(LeaderCheckRequest request) {
        DiscoveryNodes discoveryNodes = this.discoveryNodes;
        assert (discoveryNodes != null);
        StatusInfo statusInfo = this.nodeHealthService.getHealth();
        if (statusInfo.getStatus() == StatusInfo.Status.UNHEALTHY) {
            String message = "rejecting leader check from [" + String.valueOf(request.getSender()) + "] since node is unhealthy [" + statusInfo.getInfo() + "]";
            logger.debug(message);
            throw new NodeHealthCheckFailureException(message, new Object[0]);
        }
        if (!discoveryNodes.isLocalNodeElectedClusterManager()) {
            logger.debug("rejecting leader check on non-cluster-manager {}", (Object)request);
            throw new CoordinationStateRejectedException("rejecting leader check from [" + String.valueOf(request.getSender()) + "] sent to a node that is no longer the cluster-manager", new Object[0]);
        }
        if (!discoveryNodes.nodeExists(request.getSender())) {
            logger.debug("rejecting leader check from removed node: {}", (Object)request);
            throw new CoordinationStateRejectedException("rejecting leader check since [" + String.valueOf(request.getSender()) + "] has been removed from the cluster", new Object[0]);
        }
        logger.trace("handling {}", (Object)request);
    }

    private void handleDisconnectedNode(DiscoveryNode discoveryNode) {
        CheckScheduler checkScheduler = this.currentChecker.get();
        if (checkScheduler != null) {
            checkScheduler.handleDisconnectedNode(discoveryNode);
        } else {
            logger.trace("disconnect event ignored for {}, no check scheduler", (Object)discoveryNode);
        }
    }

    private class CheckScheduler
    implements Releasable {
        private final AtomicBoolean isClosed = new AtomicBoolean();
        private final AtomicLong failureCountSinceLastSuccess = new AtomicLong();
        private final DiscoveryNode leader;

        CheckScheduler(DiscoveryNode leader) {
            this.leader = leader;
        }

        public void close() {
            if (!this.isClosed.compareAndSet(false, true)) {
                logger.trace("already closed, doing nothing");
            } else {
                logger.debug("closed");
            }
        }

        void handleWakeUp() {
            if (this.isClosed.get()) {
                logger.trace("closed check scheduler woken up, doing nothing");
                return;
            }
            logger.trace("checking {} with [{}] = {}", (Object)this.leader, (Object)LEADER_CHECK_TIMEOUT_SETTING.getKey(), (Object)LeaderChecker.this.leaderCheckTimeout);
            LeaderChecker.this.transportService.sendRequest(this.leader, LeaderChecker.LEADER_CHECK_ACTION_NAME, (TransportRequest)new LeaderCheckRequest(LeaderChecker.this.transportService.getLocalNode()), TransportRequestOptions.builder().withTimeout(LeaderChecker.this.leaderCheckTimeout).withType(TransportRequestOptions.Type.PING).build(), (TransportResponseHandler)new TransportResponseHandler<TransportResponse.Empty>(){

                public TransportResponse.Empty read(StreamInput in) {
                    return TransportResponse.Empty.INSTANCE;
                }

                public void handleResponse(TransportResponse.Empty response) {
                    if (CheckScheduler.this.isClosed.get()) {
                        logger.debug("closed check scheduler received a response, doing nothing");
                        return;
                    }
                    CheckScheduler.this.failureCountSinceLastSuccess.set(0L);
                    CheckScheduler.this.scheduleNextWakeUp();
                }

                public void handleException(TransportException exp) {
                    if (CheckScheduler.this.isClosed.get()) {
                        logger.debug("closed check scheduler received a response, doing nothing");
                        return;
                    }
                    if (exp instanceof ConnectTransportException || exp.getCause() instanceof ConnectTransportException) {
                        logger.debug((Message)new ParameterizedMessage("leader [{}] disconnected during check", (Object)CheckScheduler.this.leader), (Throwable)exp);
                        CheckScheduler.this.leaderFailed((Exception)new ConnectTransportException(CheckScheduler.this.leader, "disconnected during check", (Throwable)exp));
                        return;
                    }
                    if (exp.getCause() instanceof NodeHealthCheckFailureException) {
                        logger.debug((Message)new ParameterizedMessage("leader [{}] health check failed", (Object)CheckScheduler.this.leader), (Throwable)exp);
                        CheckScheduler.this.leaderFailed((Exception)((Object)new NodeHealthCheckFailureException("node [" + String.valueOf(CheckScheduler.this.leader) + "] failed health checks", exp)));
                        return;
                    }
                    long failureCount = CheckScheduler.this.failureCountSinceLastSuccess.incrementAndGet();
                    if (failureCount >= (long)LeaderChecker.this.leaderCheckRetryCount) {
                        logger.debug((Message)new ParameterizedMessage("leader [{}] has failed {} consecutive checks (limit [{}] is {}); last failure was:", new Object[]{CheckScheduler.this.leader, failureCount, LEADER_CHECK_RETRY_COUNT_SETTING.getKey(), LeaderChecker.this.leaderCheckRetryCount}), (Throwable)exp);
                        CheckScheduler.this.leaderFailed((Exception)new SkyliteException("node [" + String.valueOf(CheckScheduler.this.leader) + "] failed [" + failureCount + "] consecutive checks", (Throwable)exp, new Object[0]));
                        return;
                    }
                    logger.debug((Message)new ParameterizedMessage("{} consecutive failures (limit [{}] is {}) with leader [{}]", new Object[]{failureCount, LEADER_CHECK_RETRY_COUNT_SETTING.getKey(), LeaderChecker.this.leaderCheckRetryCount, CheckScheduler.this.leader}), (Throwable)exp);
                    CheckScheduler.this.scheduleNextWakeUp();
                }

                public String executor() {
                    return "same";
                }
            });
        }

        void leaderFailed(final Exception e) {
            if (this.isClosed.compareAndSet(false, true)) {
                LeaderChecker.this.transportService.getThreadPool().generic().execute(new Runnable(){

                    @Override
                    public void run() {
                        LeaderChecker.this.onLeaderFailure.accept(e);
                    }

                    public String toString() {
                        return "notification of leader failure: " + e.getMessage();
                    }
                });
            } else {
                logger.trace("already closed, not failing leader");
            }
        }

        void handleDisconnectedNode(DiscoveryNode discoveryNode) {
            if (discoveryNode.equals((Object)this.leader)) {
                logger.debug("leader [{}] disconnected", (Object)this.leader);
                this.leaderFailed((Exception)new NodeDisconnectedException(discoveryNode, "disconnected"));
            }
        }

        private void scheduleNextWakeUp() {
            logger.trace("scheduling next check of {} for [{}] = {}", (Object)this.leader, (Object)LEADER_CHECK_INTERVAL_SETTING.getKey(), (Object)LeaderChecker.this.leaderCheckInterval);
            LeaderChecker.this.transportService.getThreadPool().schedule(new Runnable(){

                @Override
                public void run() {
                    CheckScheduler.this.handleWakeUp();
                }

                public String toString() {
                    return "scheduled check of leader " + String.valueOf(CheckScheduler.this.leader);
                }
            }, LeaderChecker.this.leaderCheckInterval, "same");
        }
    }

    static class LeaderCheckRequest
    extends TransportRequest {
        private final DiscoveryNode sender;

        LeaderCheckRequest(DiscoveryNode sender) {
            this.sender = sender;
        }

        LeaderCheckRequest(StreamInput in) throws IOException {
            super(in);
            this.sender = new DiscoveryNode(in);
        }

        public void writeTo(StreamOutput out) throws IOException {
            super.writeTo(out);
            this.sender.writeTo(out);
        }

        public DiscoveryNode getSender() {
            return this.sender;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || ((Object)((Object)this)).getClass() != o.getClass()) {
                return false;
            }
            LeaderCheckRequest that = (LeaderCheckRequest)((Object)o);
            return Objects.equals(this.sender, that.sender);
        }

        public int hashCode() {
            return Objects.hash(this.sender);
        }

        public String toString() {
            return "LeaderCheckRequest{sender=" + String.valueOf(this.sender) + "}";
        }
    }

    public static final class SettingsProviderImpl
    implements SettingsProvider {
        public List<? extends Setting<?>> getSettings() {
            return Arrays.asList(LEADER_CHECK_TIMEOUT_SETTING, LEADER_CHECK_INTERVAL_SETTING, LEADER_CHECK_RETRY_COUNT_SETTING);
        }
    }
}

